<?php

namespace App\Http\Controllers;

use App\Models\Invitation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Http\Requests\addScanMember;
use App\Models\StaffAssignment;

class ScanController extends Controller
{
  /**
   * Display a listing of the resource.
   */
  public function index()
  {
    return view('theme.scanner.scan');
  }

  /**
   * Show the form for creating a new resource.
   */
  public function create()
  {
    return view('theme.scanner.addToTeam');
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(addScanMember $request)
  {


    $owner = Auth::user();

    $current = $owner->assignedStaff()->count();
    $limit   = (int) $owner->staff_limit;

    if ($current >= $limit) {
      return back()->with('addMemberToTeamAlert', 'وصلت للحد الأقصى من المنظمين');
    }

    $validator = $request->validated();

    $user = User::create([
      'name' => $validator['event_name'],
      'email' => $validator['email'],
      'password' => bcrypt($validator['password']),
      'events_limit' => 0,
      'user_invitation_limit' => 0,
      'role' => 'staff',
    ]);

    if ($user) {

      StaffAssignment::create([
        'owner_id' => Auth::user()->id,
        'staff_id' => $user->id,
      ]);

      return redirect()->back()->with('addMemberToTeamSuccess', 'تم إضافة المنظم بنجاح');
    }

    return redirect()->back()->with('addMemberToTeamError', 'حدث خطأ أثناء إضافة المنظم');
  }

  /**
   * Display the specified resource.
   */
  public function show(string $id) {}

  /**
   * Show the form for editing the specified resource.
   */
  public function edit(string $id) {}

  /**
   * Update the specified resource in storage.
   */
  public function update(Request $request, string $id)
  {
    //
  }

  /**
   * Remove the specified resource from storage.
   */
  public function destroy(string $id)
  {
    //
  }

  public function verifyToken(Request $request)
  {
    try {

      // 0) لازم يكون staff
      if (!in_array(Auth::user()->role, ['staff', 'stuff'], true)) {
        return response()->json(['status' => 'error', 'message' => 'غير مصرح'], 403);
      }

      $request->validate([
        'token' => ['required', 'string'],
      ]);

      $token = trim((string)$request->input('token'));

      // 1) ابحث عن الدعوة
      $invitation = Invitation::where('qr_token', $token)->first();

      if (!$invitation) {
        return response()->json([
          'status'  => 'error',
          'type'    => 'not_found',
          'message' => 'الدعوة غير موجودة أو غير صالحة',
        ], 404);
      }

      // 2) استخرج owner_id (صاحب الدعوة)
      // ✅ خيار A: لو invitations فيها user_id
      $ownerId = $invitation->user_id ?? null;

      // ✅ خيار B: لو المالك عبر event->user_id
      // (يتطلب علاقة event في Invitation: belongsTo(Event::class))
      if (!$ownerId && method_exists($invitation, 'event') && $invitation->relationLoaded('event')) {
        $ownerId = optional($invitation->event)->user_id;
      }
      if (!$ownerId && method_exists($invitation, 'event')) {
        // لو العلاقة غير محمّلة، نحمّلها
        $invitation->load('event');
        $ownerId = optional($invitation->event)->user_id;
      }

      if (!$ownerId) {
        return response()->json([
          'status'  => 'error',
          'message' => 'تعذر تحديد صاحب الدعوة',
        ], 500);
      }

      // 3) تحقق أن الـ staff مُعتمد من هذا الـ owner
      $staffId = Auth::id();

      $isAllowed = Auth::user()
        ->ownersImAssignedTo()
        ->where('owner_id', $ownerId)
        ->exists();

      if (!$isAllowed) {
        return response()->json([
          'status'  => 'error',
          'message' => 'غير مصرح لك بمسح دعوات هذه المناسبة',
        ], 403);
      }

      // 4) حالة مستخدمة؟
      if ($invitation->is_used) {
        return response()->json([
          'status'  => 'error',
          'type'    => 'used',
          'message' => 'تم استخدام هذه الدعوة مسبقًا',
          'used_at' => optional($invitation->used_at)->format('Y-m-d H:i:s'),
          'scanned_by' => $invitation->scanned_by,
        ], 200);
      }

      // 5) حدّث الدعوة كمستخدمة
      $invitation->update([
        'is_used'    => true,
        'used_at'    => now(),
        'scanned_by' => $staffId,
      ]);

      return response()->json([
        'status'  => 'success',
        'type'    => 'valid',
        'message' => 'تم التحقق من الدعوة بنجاح ✅',
      ], 200);
    } catch (\Throwable $e) {
      return response()->json([
        'status'  => 'error',
        'message' => 'Server error: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function showTeamMembers()
  {


    $teamMembers = StaffAssignment::with('staff')
      ->where('owner_id', Auth::id())
      ->paginate(5)
      ->through(fn($item) => $item->staff);

    return view('theme.scanner.organizers', compact('teamMembers'));
  }
}
