<?php

namespace App\Http\Controllers;

use App\Models\Event;
use App\Models\User;
use App\Models\Invitation;
use App\Http\Requests\EditEventRequest;
use Illuminate\Support\Facades\Storage;
use App\Http\Requests\AddUserRequest;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Password;


class DashboardController extends Controller
{

  // Home page returning statistics
  public function index()
  {
    $stats = [
      ['label' => 'إجمالي المناسبات', 'value' => Event::count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'إجمالي الدعوات المرسلة', 'value' => Invitation::count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'عدد المنظمين', 'value' => User::where('role', 'staff')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
    ];


    return view('theme.dashboard.home', compact('stats'));
  }

  // Accounts page returning accounts
  public function accounts()
  {
    $stats = [
      ['label' => 'إجمالي الحسابات', 'value' => User::count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'صاحب مناسبة', 'value' => User::where('role', 'admin')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'حسابات المنظمين',   'value' => User::where('role', 'staff')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'حسابات المسؤول', 'value' => User::where('role', 'super-admin')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
    ];

    $eventOwners = User::where('role', '=', 'admin')->paginate(5);
    return view('theme.dashboard.accounts', compact('stats', 'eventOwners'));
  }

  public function events(User $user)
  {

    $events = $user->events()->latest()->paginate(5);
    return view('theme.dashboard.events', compact('events', 'user'));
  }

  public function editEvent(User $user, Event $event)
  {
    abort_if($event->user_id !== $user->id, 404);
    return view('theme.dashboard.editEvent', compact('event', 'user'));
  }

  public function updateEvent(EditEventRequest $request, User $user, Event $event)
  {

    $event = $user->events()->findOrFail($event->id);

    $data = $request->validated();

    if ($request->hasFile('event_image')) {

      if (!empty($event->event_image) && Storage::disk('public')->exists('events/img/' . $event->event_image)) {
        Storage::disk('public')->delete('events/img/' . $event->event_image);
      }

      $file = $request->file('event_image');
      $fileName = time() . "_" . base64_encode($file->getClientOriginalName()) . "." . $file->getClientOriginalExtension();

      $file->storeAs('events/img', $fileName, 'public');
      $data['event_image'] = $fileName;
    } else {
      unset($data['event_image']);
    }

    if ($event->update($data)) {
      return redirect()
        ->route('dashboard.editEvent', ['user' => $user->id, 'event' => $event->id])
        ->with('eventSuccess', 'تم تحديث المناسبة بنجاح');
    } else {
      return redirect()
        ->route('dashboard.editEvent', ['user' => $user->id, 'event' => $event->id])
        ->with('eventError', 'فشل تحديث المناسبة');
    }
  }
  public function deleteEvent(User $user, Event $event)
  {
    abort_if($event->user_id !== $user->id, 404);
    if ($event->delete()) {
      return redirect()
        ->route('dashboard.events', ['user' => $user->id])
        ->with('eventSuccess', 'تم حذف المناسبة بنجاح');
    } else {
      return redirect()
        ->route('dashboard.events', ['user' => $user->id])
        ->with('eventError', 'فشل حذف المناسبة');
    }
  }

  // Users logic

  public function addUser()
  {
    return view('theme.dashboard.addUser');
  }

  public function storeUser(AddUserRequest $request)
  {
    $data = $request->validated();

    $data['phone_number'] = $request->country_code . $request->phone_number;
    $data['role'] = 'admin';

    // كلمة مرور مؤقتة/عشوائية (لن يعرفها المستخدم)
    $data['password'] = Str::random(32);

    $user = User::create($data);

    // إنشاء توكن وإرسال رابط "تعيين كلمة المرور"
    $token = Password::createToken($user);
    $user->sendPasswordResetNotification($token);

    return redirect()
      ->route('dashboard.accounts')
      ->with('userSuccess', 'تم إضافة المستخدم وإرسال رابط تعيين كلمة المرور إلى بريده.');
  }
}
